/** @file   playercontroller.h
 * @brief   Declaration of PlayerController - class.
 * @version $Revision: 1.2 $
 * @author  Tomi Lamminsaari
 */

#ifndef H_WWW_PLAYERCONTROLLER_H
#define H_WWW_PLAYERCONTROLLER_H
#include "basecontroller.h"

namespace WeWantWar {

/** @class  PlayerController
 * @brief   The controller used by the Player
 * @author  Tomi Lamminsaari
 *
 * This controller reads both the keyboard and mouse input to control the
 * Player - object. The keys being used are defined in @c Settings - class.
 */
class PlayerController : public BaseController
{
public:

  ///
  /// Constants, datatype and static members
  /// ======================================
  
  /** The x-coordinate of the sight */
  int m_sightX;
  /** The y-coordinate of the sight */
  int m_sightY;
  /** Indicates that player should throw a grenade. */
  int m_grenade;
  /** Indicates if player wants to detonate the grenades */
  int m_detonate;
  
  /** Indicates that player should change weapon. -1 if no weaponchange */
  int m_weaponChange;
  

  ///
  /// Constructors, destructor and operators
  /// ======================================

    /** Constructor.
     * @param     pObj              Pointer to player
     */
	PlayerController( GameObject* pObj );


	/** Destructor
     */
	virtual ~PlayerController();

private:

    /** Copy constructor.
     * @param     rO                Reference to another PlayerController
     */
    PlayerController( const PlayerController& rO );

	/** Assignment operator
     * @param     rO                Reference to another PlayerController
     * @return    Reference to us.
     */
    PlayerController& operator = ( const PlayerController& rO );

public:


  ///
  /// Methods
  /// =======

  /** Updates this controller. Reads the keyboard/mouse and sets the control
   * flags accordingly.
   */
  virtual void update();
  
  /** Resets this controller.
   */
  virtual void reset();
  
  /** Returns a reference to member that tells if player has pressed
   * one of the "change weapon" keys.
   * @return    Reference to m_weaponChange - member.
   */
  inline int& weaponChange() { return m_weaponChange; }


  ///
  /// Getter methods
  /// ==============

  /** Returns the crosshair pointer's x-coordinate.
   * @return    Crosshair's x-coordinate.
   */
  inline int sightX() const { return m_sightX; }
  
  /** Returns the crosshair pointer's y-coordinate
   * @return    Crosshair's y-coordinate.
   */
  inline int sightY() const { return m_sightY; }
  
  /** Tells if player has pressed <i>Detonate grenades</i> - key.
   * @return    Nonzero if player has pressed detonate-key.
   */
  inline int detonate() const { return m_detonate; }
  
  /** Tells if player has pressed <i>Throw grenade</i> - key.
   * @return    Nonzero if player has pressed grenade-key.
   */
  inline int grenade() const { return m_grenade; }
  
  /** Tells if player has pressed <i>Change weapon</i> - key.
   * @return    0 if not pressed. Otherwise the weapon number.
   */
  inline int weaponChange() const { return m_weaponChange; }

protected:

  /** Scales the mouse movements in relative mode so that the player does
   * not turn too quick.
   * @param     pX                Pointer to x-mickey. New value is retuned
   *                              through the same pointer.
   * @param     pY                Pointer to y-mickey. New value is retuned
   *                              through the same pointer.
   */
  void scaleRelativeMouse( int* pX, int* pY ) const;
  
  
  
  ///
  /// Members
  /// =======



private:

  ///
  /// Private members
  /// ===============
  int iPreviousWeapon;
};

};  // end of namespace

#endif

/**
 * Version history
 * ===============
 * $Log: playercontroller.h,v $
 * Revision 1.2  2006/08/20 21:12:28  lamminsa
 * Support for mousewheel added.
 *
 * Revision 1.1.1.1  2006/01/21 23:02:42  lamminsa
 * no message
 *
 * Revision 1.0  2005-11-06 01:17:09+02  lamminsa
 * Initial revision
 *
 */
 
